/**
 * \file: mspin_demo_appl.cpp
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * MySpin 
 *
 * \component: MSPIN
 *
 * \author: Bui Le Thuan / thuan.buile@vn.bosch.com
 *          Torsten Plate ICT-ADITG/SW2 tplate@de.adit-jv.com
 *
 * \copyright: (c) 2003 - 2013 ADIT Corporation
 *
 * \history
 * 0.1 TPlate Initial version
 *
 ***********************************************************************/
#include "mspin_demo_monitor.h"
#include <string>
#include <mspin_demo_types.h>

#include <mspin_demo_timer.h>
#include <mspin_logging.h>

#ifndef MSPIN_DEMO_TOUCH_DISABLE
#include <mspin_demo_touch.h>
#endif //#ifndef MSPIN_DEMO_TOUCH_DISABLE

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <getopt.h>
#include <string.h>

#include <sys/signalfd.h>
#include <errno.h>
#include <sys/stat.h>

#ifndef MSPIN_DEMO_AUDIO_DISABLE
#include <mspin_audio_stream.h>
#include <mspin_vr_stream.h>
#endif //#ifndef MSPIN_DEMO_AUDIO_DISABLE
#include <iostream>

#include <glib-unix.h>
#include "mspin_demo_testbed.h"


//To enable app launch popup (alert) on iPhone, define MSPIN_DEMO_SHOW_APP_LAUNCH_POPUP. If not defined, no
// popup will be shown. mySPIN launcher will be directly started
//#define MSPIN_DEMO_SHOW_APP_LAUNCH_POPUP
#define MSPIN_KEY_HOME_SIGNO    10
#define MSPIN_KEY_BACK_SIGNO    16
#define MSPIN_KEY_MENU_SIGNO    30
#define MSPIN_KEY_SEARCH_SIGNO  31
#define MSPIN_END_CHAR          'E'

const struct option longopts[] =
{
  {"audio",        required_argument,  0, 'a'},
  {"boardtype",    required_argument,  0, 'b'},
  {"sink",         required_argument,  0, 'c'},
  {"testbed",      no_argument,        0, 'd'},
  {"layerid",      required_argument,  0, 'i'},
  {"compression",  required_argument,  0, 'k'},
  {"width",        required_argument,  0, 'w'},
  {"height",       required_argument,  0, 'u'},
  {"frame",        no_argument,        0, 'f'},
  {"timestamp",    no_argument,        0, 't'},
  {"verbose",      required_argument,  0, 'v'},
  {"help",         no_argument,        0, 'h'},
  {"gst-debug",    optional_argument,  0, 'g'},
  {"gst*",         optional_argument,  0, 'g'},
  {"performance",  no_argument,        0, 'p'},
  {"source",       required_argument,  0, 'r'},
  {"smoketest",    optional_argument,  0, 's'},
  {"wifiConnection",no_argument,       0, 'x'},
  {0,0,0,0},
};

#ifndef MSPIN_DEMO_AUDIO_DISABLE
static MySpinAudioStream* mySpinAudioStream = NULL;
#endif //#ifndef MSPIN_DEMO_AUDIO_DISABLE

static bool gMSPINDEMO_run = TRUE;
static bool gLayer1inUse = FALSE;

#ifndef MSPIN_DEMO_TOUCH_DISABLE
static pthread_t gTouchHandleThread = 0;
static bool gEnableTouchFields = TRUE;
#endif //#ifndef MSPIN_DEMO_TOUCH_DISABLE

static U32 gEnableAudio = 0; //Variable is S32 because read from stdin. Change here for default behavior. See also help
static bool gBoardTypeSD = FALSE;
static bool gEnablePerformanceMeasurements = FALSE;
static bool gEnableFrameMeasurements = FALSE;
static mspin_demo_instanceParameter_t gMyspinInstances[2]; //we have at max 2 mySPIN instances
static U32 gSmoketestTimeout = 0; //0 means no smoketest mode
static U32 gDemoTimeout = 0; //0 means no timeout
static char gAudioSource[64] = MYSPIN_DEMO_APPL_AUDIO_SOURCE_DEFAULT_APPLE_DEVICE;
static bool gDetectAudioSource = TRUE;
static char gAudioSink[64] = MYSPIN_DEMO_APPL_AUDIO_SINK_DEFAULT;
static U32 gLayer1Id = MSPIN_DEMO_APPL_MYSPIN1_LAYER_ID;
static U32 gLayer1Width = MSPIN_DEMO_APPL_MYSPIN1_WIDTH;
static U32 gLayer1Height = MSPIN_DEMO_APPL_MYSPIN1_HEIGHT;
static MSPIN_FRAME_COMPRESSION gFrameCompression = MSPIN_DEMO_APPL_MYSPIN_FRAME_COMP;
static bool gWifiConnect = FALSE;
static int    gEndCharFd[2];
static mspin_demo_return_state_t gDemoAppState = eMSPIN_RETURN_STATE_NO_DEVICE;     // currently support for only 1 connection

#ifndef MSPIN_DEMO_TOUCH_DISABLE
static DemoTouch demoTouch;
#endif //#ifndef MSPIN_DEMO_TOUCH_DISABLE


// using to check status of session
MSPIN_ERROR g_session_status = MSPIN_ERROR_UNKNOWN;
std::mutex g_session_monitor_m;
std::condition_variable g_session_monitor_cv;
bool g_running_session = false;

// using in the mspin_demo_monitor instance
mspin_demo_monitor_context_t monitorContext;

#ifndef MSPIN_DEMO_AUDIO_DISABLE
static void mspin_demo_findAudioSource(void)
{
    //Limitation: This algorithm only works for one Android device. In case there are two connected,
    // devices we cannot differentiate because vendorID and productID are the Google AOAP ones

    S32 i = 0;
    size_t j = 0;
    char cardXPath[64];
    char usbidToSearch[10];
    char usbidValue[20];
    char idValue[64];
    char usbidPath[64];
    char idPath[64];
    int usbidFd;
    int idFd;
    struct stat st;

    snprintf(usbidToSearch, 9, "%x:%x", MSPIN_DEMO_APPL_GOOGLE_VENDOR_ID,
            MSPIN_DEMO_APPL_GOOGLE_PRODUCT_ID_MIN);

    mspin_log_printLn(eMspinVerbosityDebug,
            "%s() search for usbid='%s'",
            __FUNCTION__, usbidToSearch);

    //Check all cardX directories in /proc/asound/
    for (i = 0; i < 8; i++)
    {
        snprintf(cardXPath, 64, "/proc/asound/card%d", i);

        if (stat(cardXPath, &st) == 0)
        {
            mspin_log_printLn(eMspinVerbosityVerbose, "%s() directory %s exists",
                    __FUNCTION__, cardXPath);

            snprintf(usbidPath, 64, "%s/usbid", cardXPath);
            usbidFd = open(usbidPath, O_RDONLY);
            if (-1 != usbidFd)
            {
                if (0 < read(usbidFd, usbidValue, sizeof(usbidValue)))
                {
                    if (0 == strncmp(usbidToSearch, usbidValue, 8)) //compare only the first 8 chars (18d1:2d0). The last char can vary between 2-5
                    {
                        mspin_log_printLn(eMspinVerbosityVerbose,
                                "%s() matching usbid found at '%s'",
                                __FUNCTION__, cardXPath);

                        //Correct USB sound device found
                        snprintf(idPath, 64, "%s/id", cardXPath);
                        idFd = open(idPath, O_RDONLY);
                        if (-1 != idFd)
                        {
                            if (0 < read(idFd, idValue, sizeof(idValue)))
                            {
                                //Remove the line break from the end
                                for (j=0; j < strlen(idValue); j++)
                                {
                                    if ('\n' == idValue[j])
                                    {
                                        idValue[j] = '\0';
                                    }
                                }

                                sprintf(gAudioSource, "hw:%s,%d", idValue, 0); //using always device 0

                                mspin_log_printLn(eMspinVerbosityInfo,
                                        "%s() matching usbid found at '%s' => using id='%s'",
                                        __FUNCTION__, cardXPath, gAudioSource);
                            }
                            else
                            {
                                mspin_log_printLn(eMspinVerbosityError,
                                        "%s() ERROR: Failed to read from file='%s' errno='%s'",
                                        __FUNCTION__, idPath, strerror(errno));
                            }
                            close(idFd);
                        }
                        else
                        {
                            mspin_log_printLn(eMspinVerbosityError,
                                    "%s() ERROR: Failed to open file='%s' errno='%s'",
                                    __FUNCTION__, idPath, strerror(errno));
                        }

                        //Alternative: Use only the card index instead of the id
//                        sprintf(gAudioSource, "hw:%d,%d", i, 0); //using found card i and device 0
//                        mspin_log_printLn(eMspinVerbosityInfo,
//                                "%s() matching usbid found at '%s' => using id='%s'",
//                                __FUNCTION__, cardXPath, gAudioSource);
                    }
                    else
                    {
                        mspin_log_printLn(eMspinVerbosityInfo,
                                "%s() value of '%s' is '%s' but does not match",
                                __FUNCTION__, usbidPath, usbidValue);
                    }
                }
                else
                {
                    mspin_log_printLn(eMspinVerbosityError,
                            "%s(ERROR: Failed to read from file='%s' errno='%s'",
                            __FUNCTION__, usbidPath, strerror(errno));
                }
                close(usbidFd);
            }
            else
            {
                if (ENOENT == errno)
                {
                    mspin_log_printLn(eMspinVerbosityDebug,
                            "%s() Failed to open file='%s' errno='%s'",
                            __FUNCTION__, usbidPath, strerror(errno));
                }
                else
                {
                    mspin_log_printLn(eMspinVerbosityError,
                            "%s() ERROR: Failed to open file='%s' errno='%s'",
                            __FUNCTION__, usbidPath, strerror(errno));
                }
            }
        }
    }
}
#endif //#ifndef MSPIN_DEMO_AUDIO_DISABLE

static void mspin_demo_switchToStop(mspin_demo_connectionParameter_t *pDevice)
{
    if (!pDevice)
    {
        return;
    }

    if (pDevice->audioSupport > 0)
    {
#ifndef MSPIN_DEMO_AUDIO_DISABLE
        mspin_log_printLn(eMspinVerbosityInfo,
                "%s(device=%p) stop audio playback",
                __FUNCTION__, pDevice);

        //ToDo: Change from global static to the member of mspin_demo_connectionParameter_t.
        mySpinAudioStream->stopAudioStream();
        delete mySpinAudioStream;
        mySpinAudioStream = NULL;

#else
        mspin_log_printLn(eMspinVerbosityWarn,
                "%s(device=%p)WARNING: compile without MSPIN_DEMO_AUDIO_DISABLE",
                __FUNCTION__, pDevice);
#endif //#ifndef MSPIN_DEMO_AUDIO_DISABLE
    }

    if (DeviceMonitor::instance().isIOSDevice(pDevice->vendorId, pDevice->productId)) //iOS devices only
    {
        MSPIN_EAPSessionStop(pDevice->pMyspin);
        pDevice->runState = eMSPIN_DEMO_STATE_STOPPED;

        mspin_log_printLn(eMspinVerbosityDebug,
                "%s(device=%p) switching to STOP is complete",
                __FUNCTION__, pDevice);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityDebug,
                "%s(device=%p) AOAP case -> do nothing",
                __FUNCTION__, pDevice);
    }
}

static MSPIN_ERROR mspin_demo_switchToRun(mspin_demo_connectionParameter_t *pDevice)
{
    MSPIN_ERROR result = MSPIN_SUCCESS;

    if (!pDevice)
    {
        return MSPIN_ERROR_GENERAL;
    }

    if (pDevice->audioSupport > 0)
    {
#ifndef MSPIN_DEMO_AUDIO_DISABLE

        audioStreamDeviceType audioType = audioStreamDeviceType::AUDIO_DEVICE_TYPE_ALSA;

        if (gDetectAudioSource &&  (pDevice->audioSupport>=1) )
        {
            if(DeviceMonitor::instance().isIOSDevice(pDevice->vendorId, pDevice->productId))
            {
                // AudioSource is hardcoded UAC2Gadget
                strcpy(gAudioSource, MYSPIN_DEMO_APPL_AUDIO_SOURCE_DEFAULT_APPLE_DEVICE);
                audioType = audioStreamDeviceType::AUDIO_DEVICE_TYPE_ALSA_W_SNDCTL;
            }
            else
            {
                // Find the Audio Source
                mspin_demo_findAudioSource();
                audioType = audioStreamDeviceType::AUDIO_DEVICE_TYPE_ALSA;
            }

            mspin_log_printLn(eMspinVerbosityVerbose,
                            "%s: vendorId = %.4x, productId = %.4x, gAudioSource = %s, audioType = %d", __FUNCTION__,
                            pDevice->vendorId, pDevice->productId, gAudioSource, audioType);
        }

        //ToDo: Change from global static pointer to the member of mspin_demo_connectionParameter_t.
        mySpinAudioStream = new MySpinAudioStream();

        mspin_log_printLn(eMspinVerbosityInfo,
                "%s(device=%p) start mySPINAudioStreamLib",
                __FUNCTION__, pDevice);

        S32 setAudioStrIn;
        setAudioStrIn = mySpinAudioStream->setAudioStreamInputDevice(gAudioSource, audioType);
        mspin_log_printLn(eMspinVerbosityInfo,
                "%s setAudioStreamInputDevice returns: %d  ",
                __FUNCTION__, setAudioStrIn);

        S32 setAudioStrOut;
        setAudioStrOut = mySpinAudioStream->setAudioStreamOutputDevice(MYSPIN_DEMO_APPL_AUDIO_SINK_DEFAULT);
        mspin_log_printLn(eMspinVerbosityInfo,
                "%ssetAudioStreamOutputDevice returns: %d  ",
                __FUNCTION__, setAudioStrOut);

        audioStreamConfig aStreamConf;
        aStreamConf.codecName = "";
        aStreamConf.sampleRate = 44100;
        aStreamConf.channels = 2;
        aStreamConf.sampleDepth = 16;
        aStreamConf.sampleWidth = 16;
        aStreamConf.sampleSignedness = TRUE;
        aStreamConf.sampleEndianness = FALSE;

        S32 setAudioconf;
        setAudioconf = mySpinAudioStream->setAudioStreamConfig(aStreamConf);
        mspin_log_printLn(eMspinVerbosityInfo,
                "%s setAudioStreamConfig returns: %d",
                __FUNCTION__, setAudioconf);

        S32 setAudioStart;
        setAudioStart = mySpinAudioStream->startAudioStream();
        mspin_log_printLn(eMspinVerbosityInfo,
                "%s startAudioStream returns: %d",
                __FUNCTION__, setAudioStart);

#else
        mspin_log_printLn(eMspinVerbosityWarn,
                "%s(device=%p)WARNING: compile without MSPIN_DEMO_AUDIO_DISABLE",
                __FUNCTION__, pDevice);
#endif //#ifndef MSPIN_DEMO_AUDIO_DISABLE
    }

    if (DeviceMonitor::instance().isIOSDevice(pDevice->vendorId, pDevice->productId)) //iOS devices only
    {
        result = MSPIN_EAPSessionStart(pDevice->pMyspin);
        if (MSPIN_SUCCESS == result)
        {
            mspin_log_printLn(eMspinVerbosityDebug,
                    "%s(device=%p) Switching to RUN is complete",
                    __FUNCTION__, pDevice);
            gDemoAppState = eMSPIN_RETURN_STATE_SUCCESSFUL_CONNECTION;
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s(device=%p) ERROR: Switching to RUN failed (with '%s'(%d))",
                    __FUNCTION__, pDevice, MSPIN_GetErrorName(result), result);
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityVerbose,
                "%s(device=%p) Nothing to do for Android",
                __FUNCTION__, pDevice);
    }

    if (MSPIN_SUCCESS == result)
    {
        U32 keys = MSPIN_GetAvailableKeys(pDevice->pMyspin);
        mspin_log_printLn(eMspinVerbosityInfo,
                "%s(device=%p) keys are %d",
                __FUNCTION__, pDevice, keys);

#ifdef MYSPIN_DEMO_APPL_SEND_TEST_POSITION
        mspin_log_printLn(eMspinVerbosityInfo,
                        "%s(device=%p) sending now test position",
                        __FUNCTION__, pDevice);

        MSPIN_SendPosition(pDevice->pMyspin, "$GPRMC,113222.22,A,5209.023,N,00955.844,E,,,310114,,E");
#endif //#ifdef MYSPIN_DEMO_APPL_SEND_TEST_POSITION
    }

    pDevice->runState = eMSPIN_DEMO_STATE_RUNNING;

    return result;
}

static void mspin_demo_signalHandler(int signo)
{
    if (SIGINT == signo || SIGQUIT == signo || SIGALRM == signo)
    {
        mspin_log_printLn(eMspinVerbosityInfo,
                "%s() Signal %d received",
                __FUNCTION__, signo);

        gMSPINDEMO_run = FALSE;
        monitorContext.gQuit = TRUE;
        sem_post(&monitorContext.gConnectSemaphore); //release semaphore

        std::unique_lock<std::mutex> lk(monitorContext.g_wifi_monitor_m);
        monitorContext.g_connected_device = true;
        monitorContext.g_disconnected_device = true;
        g_running_session = true;
        lk.unlock();

        monitorContext.g_wifi_monitor_cv.notify_all();
        g_session_monitor_cv.notify_all();

        if(Testbed::instance().getTestbedMode())
        {
            Testbed::instance().adit_testbed_deinit();
        }
    }
    else
    {
        if (gMyspinInstances[0].pConnection)
        {
            if (gMyspinInstances[0].pConnection->pMyspin)
            {
                switch (signo)
                {
                    case MSPIN_KEY_HOME_SIGNO:
                        mspin_log_printLn(eMspinVerbosityInfo, "%s() Signal %d received -> send home button" ,
                                __FUNCTION__, signo);
                        MSPIN_KeyEvent(gMyspinInstances[0].pConnection->pMyspin, MSPIN_KEY_HOME, true);
                        MSPIN_KeyEvent(gMyspinInstances[0].pConnection->pMyspin, MSPIN_KEY_HOME, false);
                        break;
                    case MSPIN_KEY_BACK_SIGNO:
                        mspin_log_printLn(eMspinVerbosityInfo, "%s() Signal %d received -> send back button" ,
                                __FUNCTION__, signo);
                        MSPIN_KeyEvent(gMyspinInstances[0].pConnection->pMyspin, MSPIN_KEY_BACK, true);
                        MSPIN_KeyEvent(gMyspinInstances[0].pConnection->pMyspin, MSPIN_KEY_BACK, false);
                        break;
                    case MSPIN_KEY_MENU_SIGNO:
                        mspin_log_printLn(eMspinVerbosityInfo, "%s() Signal %d received -> send menu button" ,
                                __FUNCTION__, signo);
                        MSPIN_KeyEvent(gMyspinInstances[0].pConnection->pMyspin, MSPIN_KEY_MENU, true);
                        MSPIN_KeyEvent(gMyspinInstances[0].pConnection->pMyspin, MSPIN_KEY_MENU, false);
                        break;
                    case MSPIN_KEY_SEARCH_SIGNO:
                        mspin_log_printLn(eMspinVerbosityInfo, "%s() Signal %d received -> send search button" ,
                                __FUNCTION__, signo);
                        MSPIN_KeyEvent(gMyspinInstances[0].pConnection->pMyspin, MSPIN_KEY_SEARCH, true);
                        MSPIN_KeyEvent(gMyspinInstances[0].pConnection->pMyspin, MSPIN_KEY_SEARCH, false);
                        break;
                    default:
                        mspin_log_printLn(eMspinVerbosityError, "%s() Read unexpected signal", __FUNCTION__);
                        break;
                }
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Signal %d received but no mySPIN instance" ,
                        __FUNCTION__, signo);
            }
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Signal %d received but no mySPIN connection" ,
                    __FUNCTION__, signo);
        }
    }
}

static void mspin_demo_onPTTAvailable_CB(void* context, bool pttAvailable)
{
    mspin_log_printLn(eMspinVerbosityInfo,
                        "%s(context=%p) is PTT Available = %d ",
                        __FUNCTION__, context, pttAvailable);
}

static void mspin_demo_onError_CB(void* context, MSPIN_ERROR error)
{
    mspin_demo_connectionParameter_t *pDevice = (mspin_demo_connectionParameter_t*)context;
    if (pDevice)
    {
        mspin_demo_switch_run_state_t switchRunState = pDevice->switchRunState;
        if (eMSPIN_DEMO_STATE_STABLE != switchRunState)
        {
            mspin_log_printLn(eMspinVerbosityInfo,
                    "%s(context=%p, error='%s'(%d)) in switching state='%s' (run state='%s') -> ignore errors during switching",
                    __FUNCTION__, context, MSPIN_GetErrorName(error), error,
                    DeviceMonitor::getSwitchRunState(switchRunState).c_str(),
                    DeviceMonitor::getRunState(pDevice->runState).c_str());
        }
        else
        {
            //Switch to stop only when not already switching to another state
            pthread_mutex_lock(&(pDevice->switchStateMutex));
            if (DeviceMonitor::instance().isIOSDevice(pDevice->vendorId, pDevice->productId))
            {
                if (eMSPIN_DEMO_STATE_RUNNING == pDevice->runState)
                {
                    mspin_log_printLn(eMspinVerbosityError,
                            "%s(context=%p, error='%s'(%d)) ERROR: Switching iOS device to stop from state='%s'",
                            __FUNCTION__, context, MSPIN_GetErrorName(error), error,
                            DeviceMonitor::getRunState(pDevice->runState).c_str());
                    pDevice->switchRunState = eMSPIN_DEMO_STATE_SWITCH_TO_STOP;
                }
                else
                {
                    mspin_log_printLn(eMspinVerbosityError,
                            "%s(context=%p, error='%s'(%d)) ERROR: Switching to stop isn't required because we are already in state='%s'",
                            __FUNCTION__, context, MSPIN_GetErrorName(error), error,
                            DeviceMonitor::getRunState(pDevice->runState).c_str());
                }
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityError,
                        "%s(context=%p, error='%s'(%d)) ERROR: Terminating AOAP connection in state='%s'",
                        __FUNCTION__, context, MSPIN_GetErrorName(error), error,
                        DeviceMonitor::getRunState(pDevice->runState).c_str());
                pDevice->switchRunState = eMSPIN_DEMO_STATE_SWITCH_TO_EXIT;
            }
            pthread_mutex_unlock(&(pDevice->switchStateMutex));
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(context=%p, error='%s'(%d)) ERROR: Could not get device",
                __FUNCTION__, context, MSPIN_GetErrorName(error), error);
    }
}


static void mspin_demo_handleEvents(mspin_demo_instanceParameter_t *pMyspinInstance)
{
//    U32 cnt = 0;
    bool terminateLoop = FALSE;
    U32 waitCount = 0;

    mspin_demo_connectionParameter_t *pConnection = NULL;

    if (!pMyspinInstance)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(instance=%p) FATAL ERROR: myspin instance is NULL",
                __FUNCTION__, pMyspinInstance);
        return;
    }

    pConnection = pMyspinInstance->pConnection;
    if (!pConnection)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(instance=%p) FATAL ERROR: connection is NULL",
                __FUNCTION__, pMyspinInstance);
        return;
    }

    if (!pConnection->pMyspin)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(instance=%p) [id=%lu] FATAL ERROR: mspin is NULL",
                __FUNCTION__, pMyspinInstance, pMyspinInstance->threadId);
        return;
    }

    mspin_log_printLn(eMspinVerbosityVerbose,
            "%s(instance=%p) [id=%lu] started in run_state='%s' and switch_run_state='%s'",
            __FUNCTION__, pMyspinInstance, pMyspinInstance->threadId,
            DeviceMonitor::getRunState(pConnection->runState).c_str(),
            DeviceMonitor::getSwitchRunState(pConnection->switchRunState).c_str());

    while (!terminateLoop)
    {
        if (!gMSPINDEMO_run || (eMSPIN_DEMO_STATE_SWITCH_TO_EXIT == pConnection->switchRunState))
        {
            mspin_log_printLn(eMspinVerbosityDebug,
                    "%s(instance=%p) [id=%lu] exit this run loop (switch state='%s')",
                    __FUNCTION__, pMyspinInstance, pMyspinInstance->threadId,
                    DeviceMonitor::getSwitchRunState(pConnection->switchRunState).c_str());

            if (eMSPIN_DEMO_STATE_RUNNING == pConnection->runState)
            {
                pthread_mutex_lock(&(pConnection->switchStateMutex));
                pConnection->switchRunState = eMSPIN_DEMO_STATE_SWITCH_TO_STOP;
                pthread_mutex_unlock(&(pConnection->switchStateMutex));
            }
            terminateLoop = TRUE;
        }

        pthread_mutex_lock(&(pConnection->switchStateMutex));
        switch (pConnection->switchRunState)
        {
            case eMSPIN_DEMO_STATE_STABLE:
            {
                //Nothing to do
                break;
            }
            case eMSPIN_DEMO_STATE_SWITCH_TO_RUN:
            {
                mspin_log_printLn(eMspinVerbosityDebug,
                        "%s(instance=%p) [id=%lu] switch to run state from state='%s'",
                        __FUNCTION__, pMyspinInstance, pMyspinInstance->threadId,
                        DeviceMonitor::getRunState(pConnection->runState).c_str());

                //If mySPIN is still running, stop it before re-starting it
                if (eMSPIN_DEMO_STATE_RUNNING == pConnection->runState)
                {
                    mspin_demo_switchToStop(pConnection);
                }

                MSPIN_ERROR result = mspin_demo_switchToRun(pConnection);
                if (MSPIN_SUCCESS != result)
                {
                    mspin_log_printLn(eMspinVerbosityError,
                            "%s(instance=%p) [id=%lu] ERROR: Switching to RUN failed (with '%s'(%d)) -> revert to stop",
                            __FUNCTION__, pMyspinInstance, pMyspinInstance->threadId,
                            MSPIN_GetErrorName(result), result);

                    //In case of error switch back to stop state
                    mspin_demo_switchToStop(pConnection);
                }

                pConnection->switchRunState = eMSPIN_DEMO_STATE_STABLE;
                break;
            }
            case eMSPIN_DEMO_STATE_SWITCH_TO_STOP:
            {
                mspin_log_printLn(eMspinVerbosityInfo,
                        "%s(instance=%p) [id=%lu] switch to stop state from state='%s'",
                        __FUNCTION__, pMyspinInstance, pMyspinInstance->threadId,
                        DeviceMonitor::getRunState(pConnection->runState).c_str());

                mspin_demo_switchToStop(pConnection);

                pConnection->switchRunState = eMSPIN_DEMO_STATE_STABLE;
                break;
            }
            case eMSPIN_DEMO_STATE_SWITCH_TO_EXIT:
            {
                mspin_log_printLn(eMspinVerbosityInfo,
                        "%s(instance=%p) [id=%lu] switch to exit -> leave loop",
                        __FUNCTION__, pMyspinInstance, pMyspinInstance->threadId);
                terminateLoop = TRUE;
                pConnection->switchRunState = eMSPIN_DEMO_STATE_STABLE;
                break;
            }
            default:
            {
                mspin_log_printLn(eMspinVerbosityFatal,
                        "%s(instance=%p) [id=%lu] FATAL ERROR: Undefined state %d -> leave loop",
                        __FUNCTION__, pMyspinInstance, pMyspinInstance->threadId, pConnection->switchRunState);
                terminateLoop = TRUE;
                pConnection->switchRunState = eMSPIN_DEMO_STATE_STABLE;
                break;
            }
        }
        pthread_mutex_unlock(&(pConnection->switchStateMutex));

        if (terminateLoop)
        {
            break; //leave while loop
        }

        switch (pConnection->runState)
        {
            case eMSPIN_DEMO_STATE_RUNNING:
            {
                MSPIN_PumpEvents(pConnection->pMyspin, 50); // let layermanager adapter check for touch events
                // If you don't get touch events through your layermanager adapter implementation, then
                // do not call MSPIN_PumpEvents() - it might be necessary to wait a few ms to avoid CPU saturation.

//                cnt++;
                // test key pressed / other functions:
        //        if (cnt % (10*40) == 0)
        //        {
        ////            MSPIN_KeyEvent(pConnection->pMyspin, MSPIN_KEY_BACK,true);
        ////            MSPIN_KeyEvent(pConnection->pMyspin, MSPIN_KEY_BACK,false);
        //            MSPIN_SendPosition(pConnection->pMyspin, "$GPRMC,162614,A,5230.5900,N,01322.3900,E,10.0,90.0,131006,1.2,E,A*13");
        //            MSPIN_SetNightMode(pConnection->pMyspin, true);
        //            MSPIN_SetMovingStatus(pConnection->pMyspin, true);
        //        }
                break;
            }
            case eMSPIN_DEMO_STATE_STOPPED:
            {
                usleep(100000); //wait 100ms

                waitCount++;
                if(waitCount == 20) // 2 seconds
                {
                    pConnection->switchRunState = eMSPIN_DEMO_STATE_SWITCH_TO_EXIT;
                    g_session_status = MSPIN_ERROR_CONNECTION_TIMEOUT;
                }

                break;
            }
            default:
            {
                mspin_log_printLn(eMspinVerbosityFatal,
                        "%s(instance=%p) [id=%lu] FATAL ERROR: Undefined state -> leave loop",
                        __FUNCTION__, pMyspinInstance, pMyspinInstance->threadId);
                terminateLoop = TRUE;
                break;
            }
        }
    }

    if(Testbed::instance().getTestbedMode())
    {
        if(Testbed::instance().getConnectedAndroidPhone())
        {
            Testbed::instance().setAndroidPhoneDisconnectionStatus(TRUE);
        }
        else
        {
            Testbed::instance().setApplePhoneDisconnectionStatus(TRUE);
        }

        Testbed::instance().adit_testbed_lost();
    }
}

static void* mspin_demo_runMyspinInstance(void* param)
{
    MSPIN_ERROR conn = MSPIN_ERROR_UNKNOWN;
    bool usingLayer1 = FALSE;
    static U32 sActiveConnections = 0;
    mspin_demo_instanceParameter_t *pMyspinInstance = (mspin_demo_instanceParameter_t*)param;
    mspin_demo_connectionParameter_t *pConnection = NULL;

    if (!pMyspinInstance)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(args=%p) ERROR: Argument is NULL",
                __FUNCTION__, param);
        pthread_exit(NULL);
        return NULL;
    }

    S32 instanceIndex = -1;
    if (NULL == gMyspinInstances[0].pConnection)
    {
        instanceIndex = 0;
    }
    else if (NULL == gMyspinInstances[1].pConnection)
    {
        instanceIndex = 1;
    }

    if (0 == instanceIndex)
    {
        prctl(PR_SET_NAME,"runMyspin0",0,0,0);
    }
    else if (1 == instanceIndex)
    {
        prctl(PR_SET_NAME,"runMyspin1",0,0,0);
    }
    else
    {
        prctl(PR_SET_NAME,"runMyspinX",0,0,0);
    }

    pConnection = pMyspinInstance->pConnection;

    if (!pConnection)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(args=%p) ERROR: Connection is NULL",
                __FUNCTION__, param);

        pthread_exit(NULL);
        return NULL;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityInfo,
                "%s(args=%p) connection=%p",
                __FUNCTION__, param, pMyspinInstance->pConnection);
    }

    if (sActiveConnections > 1)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(args=%p) [id=%lu] ERROR: The maximum of mySPIN instances is already reached",
                __FUNCTION__, pMyspinInstance, pMyspinInstance->threadId);
        delete (pMyspinInstance->pConnection);
        pMyspinInstance->pConnection = NULL;
        pthread_exit(NULL);
        return NULL;
    }

    pConnection->pMyspin = MSPIN_CreateInstance(pConnection);
    if (!pConnection->pMyspin)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(args=%p) [id=%lu] ERROR: no MSPIN instance",
                __FUNCTION__, pMyspinInstance, pMyspinInstance->threadId);
        pthread_exit(NULL);
        return NULL;
    }

    ++sActiveConnections;

#ifndef MSPIN_DEMO_TOUCH_DISABLE
    //Set mySPIN instance for forwarding home key events and suspend/resume
    if (pthread_equal(pMyspinInstance->threadId, gMyspinInstances[0].threadId))
    {
        demoTouch.setInstanceHandle(pConnection->pMyspin);
    }
#endif //#ifndef MSPIN_DEMO_TOUCH_DISABLE

    if (!gLayer1inUse)
    {
        gLayer1inUse = TRUE;
        usingLayer1 = TRUE;
    }
    MSPIN_FrameProperties_t frameProps={};
    if (usingLayer1)
    {
        if(Testbed::instance().getTestbedMode())
        {
            mspin_log_printLn(eMspinVerbosityInfo, "%s - layerID = %d and surfaceID = %d.\n", __FUNCTION__,
                    Testbed::instance().getLayerID(), Testbed::instance().getSurfaceID());

            frameProps.width = 768;
            frameProps.height = 480;
            frameProps.layerId = Testbed::instance().getLayerID();
            frameProps.surfaceId = Testbed::instance().getSurfaceID();
            frameProps.format = MSPIN_PIXEL_FORMAT_RGB565;
        }
        else
        {
            frameProps.width = gLayer1Width;  // on x86sim width (of surface) must be divisible by 64 (at least with Mentor LayerMngt)
            frameProps.height = gLayer1Height;
            frameProps.layerId = gLayer1Id;
            frameProps.surfaceId = MSPIN_DEMO_APPL_MYSPIN1_SURFACE_ID;
            frameProps.format = MSPIN_PIXEL_FORMAT_RGB565;
        }

        frameProps.compression = gFrameCompression;
    }
    else
    {
        frameProps.width = MSPIN_DEMO_APPL_MYSPIN2_WIDTH;     // layer must be setup in start script
        frameProps.height = MSPIN_DEMO_APPL_MYSPIN2_HEIGHT;
        frameProps.layerId = MSPIN_DEMO_APPL_MYSPIN2_LAYER_ID;
        frameProps.surfaceId = MSPIN_DEMO_APPL_MYSPIN2_SURFACE_ID;
        frameProps.compression = gFrameCompression;
        frameProps.format = MSPIN_PIXEL_FORMAT_RGB565;
    }

    MSPIN_SetFrameProperties(pConnection->pMyspin, &frameProps);

    MSPIN_AccessoryPropertiesAoap_t accProps;
    accProps.vendor="BSOT";
    accProps.model="SUZSLN";
    accProps.description="mySPIN integration";
    accProps.version="0.1";
    accProps.uri="http://www.bosch.de";
    accProps.serial="12345";
    MSPIN_SetAccessoryPropertiesAoap(pConnection->pMyspin, &accProps);

    MSPIN_SetErrorCallback(pConnection->pMyspin, mspin_demo_onError_CB);
    MSPIN_SetCapabilities(pConnection->pMyspin, 3);
    MSPIN_EnableTouchFiltering(pConnection->pMyspin, true);

    if (!gWifiConnect)
    {
        if (DeviceMonitor::instance().isIOSDevice(pConnection->vendorId, pConnection->productId))
        {
    #ifndef MSPIN_IAP2_SUPPORT_DISABLED

            if(Testbed::instance().getTestbedMode())
            {
                while(Testbed::instance().getState() != TBA_STATE_FOREGROUND)
                {
                    mspin_log_printLn(eMspinVerbosityInfo, "%s - Waiting for changing state - iOS device.\n", __FUNCTION__);
                    sleep(1);

                    if(monitorContext.gQuit)
                    {
                        mspin_log_printLn(eMspinVerbosityInfo, "%s - Quit Application - Apple device.\n", __FUNCTION__);
                        break;
                    }
                    if(Testbed::instance().getApplePhoneDisconnectionStatus())
                    {
                        mspin_log_printLn(eMspinVerbosityInfo, "%s - Device removed - Apple device.\n", __FUNCTION__);
                        break;
                    }
                }
            }

            if(Testbed::instance().getTestbedMode() && Testbed::instance().getApplePhoneDisconnectionStatus())
            {
                //don't do anything
            }
            else
            {
                //Establish iAP2 connection and if device becomes ready call mySPIN to prepare connection
                conn = DeviceMonitor::instance().connectIAP2(pConnection);
                if (conn == MSPIN_SUCCESS)
                {
                    MSPIN_EnablePTTCustomKey(pConnection->pMyspin, true);
                    MSPIN_SetPTTAvailableCallback(pConnection->pMyspin, mspin_demo_onPTTAvailable_CB);

                    conn = MSPIN_ConnectiAP2(pConnection->pMyspin, pConnection->vendorId,
                            pConnection->productId, pConnection->serial.c_str(),
                            "/dev/ffs/ep3", "/dev/ffs/ep4", TRUE);
                }
                else
                {
                    mspin_log_printLn(eMspinVerbosityError,
                            "%s(args=%p) [id=%lu] %.4x:%.4x s/n='%s' devNum=%d ERROR: Failed to prepare iAP2 connection in mySPIN TA with %d",
                            __FUNCTION__, pMyspinInstance, pMyspinInstance->threadId,
                            pConnection->vendorId, pConnection->productId,
                            pConnection->serial.c_str(), (U32)pConnection->devNum, conn);
                }
            }

    #else
            mspin_log_printLn(eMspinVerbosityError,
                    "%s(args=%p) [id=%lu] ERROR: iAP2 isn't supported for device %.4x:%.4x s/n='%s' devNum=%d ",
                    __FUNCTION__, pMyspinInstance, pMyspinInstance->threadId,
                    pConnection->vendorId, pConnection->productId,
                    pConnection->serial.c_str(), (U32)pConnection->devNum);
            conn = MSPIN_ERROR_CONNECTION_START;
    #endif //#ifndef MSPIN_IAP2_SUPPORT_DISABLED
        }
        else
        {
            if(Testbed::instance().getTestbedMode())
            {

                while(Testbed::instance().getState() != TBA_STATE_FOREGROUND)
                {
                    mspin_log_printLn(eMspinVerbosityInfo, "%s - Waiting for changing state - Android based device.\n", __FUNCTION__);
                    sleep(1);

                    if(monitorContext.gQuit)
                    {
                        mspin_log_printLn(eMspinVerbosityInfo, "%s - Quit Application - Android based device.\n", __FUNCTION__);
                        break;
                    }
                    if(Testbed::instance().getAndroidPhoneDisconnectionStatus())
                    {
                        mspin_log_printLn(eMspinVerbosityInfo, "%s - Device removed - Android based device.\n", __FUNCTION__);
                        break;
                    }
                }
            }

            if(Testbed::instance().getTestbedMode() && Testbed::instance().getAndroidPhoneDisconnectionStatus())
            {
                //don't do anything
            }
            else
            {
                if (gEnablePerformanceMeasurements)
                {
                    MSPIN_MeasurePerformanceAOAP();
                }
                bool audioSupport = pConnection->audioSupport > 0;

                MSPIN_EnablePTTCustomKey(pConnection->pMyspin, true);
                MSPIN_SetPTTAvailableCallback(pConnection->pMyspin, mspin_demo_onPTTAvailable_CB);

                conn = MSPIN_ConnectAoap(pConnection->pMyspin, pConnection->vendorId,
                        pConnection->productId, pConnection->serial.c_str(), &audioSupport);
                if (conn == MSPIN_SUCCESS)
                {
                    gDemoAppState = eMSPIN_RETURN_STATE_SUCCESSFUL_CONNECTION;
                }
            }
        }
    }
    else
    {
        MSPIN_EnablePTTCustomKey(pConnection->pMyspin, true);
        MSPIN_SetPTTAvailableCallback(pConnection->pMyspin, mspin_demo_onPTTAvailable_CB);

        conn = MSPIN_ConnectTCPClient(pConnection->pMyspin,pConnection->connectionID);
        if (conn == MSPIN_SUCCESS)
        {
            gDemoAppState = eMSPIN_RETURN_STATE_SUCCESSFUL_CONNECTION;
        }
    }


    /* PRQA: Lint Message 788: Other error codes are not of interest and will be handled in default case */
    /*lint -save -e788*/

    g_session_status = conn;
    if( Testbed::instance().getTestbedMode() &&
            (Testbed::instance().getAndroidPhoneDisconnectionStatus()
                    || Testbed::instance().getApplePhoneDisconnectionStatus()) )
    {
        // don't do anything
    }
    else
    {
        switch (conn)
        {
            case MSPIN_SUCCESS:
            {
                mspin_log_printLn(eMspinVerbosityVerbose, "%s: Session is running", __FUNCTION__);

                mspin_demo_handleEvents(pMyspinInstance); //handle events and serve mySPIN
                break;
            }
            case MSPIN_ERROR_CONNECTION_START:
            {
                mspin_log_printLn(eMspinVerbosityError,
                        "%s(args=%p) [id=%lu] ERROR: Could not switch phone %.4x:%.4x w/s='%s' to accessory mode!",
                        __FUNCTION__, pMyspinInstance, pMyspinInstance->threadId,
                        pConnection->vendorId, pConnection->productId,
                        pConnection->serial.c_str());
                break;
            }
            case MSPIN_ERROR_PROTOCOL_SETUP:
            {
                mspin_log_printLn(eMspinVerbosityInfo,
                        "%s(args=%p) [id=%lu] Could not setup protocol with phone %.4x:%.4x w/s='%s'!",
                        __FUNCTION__, pMyspinInstance, pMyspinInstance->threadId,
                        pConnection->vendorId, pConnection->productId,
                        pConnection->serial.c_str());
                break;
            }
            case MSPIN_ERROR_CORE:
            {
                mspin_log_printLn(eMspinVerbosityInfo,
                        "%s(args=%p) [id=%lu] mySPIN Core reports error with phone %.4x:%.4x w/s='%s'!",
                        __FUNCTION__, pMyspinInstance, pMyspinInstance->threadId,
                        pConnection->vendorId, pConnection->productId,
                        pConnection->serial.c_str());
                break;
            }
            case MSPIN_ERROR_LAYERMANAGER:
            {
                mspin_log_printLn(eMspinVerbosityError,
                        "%s(args=%p) [id=%lu] ERROR: initializing layer manager!",
                        __FUNCTION__, pMyspinInstance, pMyspinInstance->threadId);
                break;
            }
            default:
            {
                mspin_log_printLn(eMspinVerbosityError,
                        "%s(args=%p) [id=%lu] ERROR: when connecting to phone %.4x:%.4x w/s='%s' with error %d!",
                        __FUNCTION__, pMyspinInstance, pMyspinInstance->threadId,
                        pConnection->vendorId, pConnection->productId,
                        pConnection->serial.c_str(), conn);
                break;
            }
        }
        /*lint -restore*/
    }

    mspin_log_printLn(eMspinVerbosityDebug,
            "%s(args=%p) [id=%lu] prepare exit -> delete mySPIN Core and Layermanager",
            __FUNCTION__, pMyspinInstance, pMyspinInstance->threadId);

    MSPIN_DeleteInstance(&(pConnection->pMyspin)); //sets it to NULL

    if(gWifiConnect)
    {
        MSPIN_CloseConnection(pConnection->connectionID);
        MSPIN_StopTCPListener();

        std::unique_lock<std::mutex> lk(monitorContext.g_wifi_monitor_m);
        monitorContext.g_disconnected_device = true;
        lk.unlock();
        monitorContext.g_wifi_monitor_cv.notify_one();

        mspin_log_printLn(eMspinVerbosityVerbose,
                "%s Device is disconnected", __FUNCTION__);

    }
    else
    {
        if(g_session_status == MSPIN_SUCCESS)
        {
            // Reset usb device list
            DeviceMonitor::instance().resetUsbDeviceList();
        }
    }

    if (usingLayer1)
    {
        gLayer1inUse = FALSE; // layer 1 not in use anymore
    }

    mspin_log_printLn(eMspinVerbosityVerbose,
            "%s(args=%p) [id=%lu] instance deleted. instance is now %p",
            __FUNCTION__, pMyspinInstance, pMyspinInstance->threadId,
            pConnection->pMyspin);

    //Do cleanup also in error cases. If something isn't initialized this shouldn't be a problem
    if (DeviceMonitor::instance().isIOSDevice(pConnection->vendorId, pConnection->productId)) //iOS devices only
    {
#ifndef MSPIN_IAP2_SUPPORT_DISABLED
        mspin_log_printLn(eMspinVerbosityDebug,
                "%s(args=%p) [id=%lu] prepare exit -> disconnect iAP2",
                __FUNCTION__, pMyspinInstance, pMyspinInstance->threadId);
        DeviceMonitor::instance().disconnectIAP2();
        MSPIN_DisconnectiAP2(NULL); //Attention: the MSPIN_Instance_t is already NULL due to the
                                    // call of 'MSPIN_DeleteInstance' but this is OK
#endif //#ifndef MSPIN_IAP2_SUPPORT_DISABLED
    }
    else
    {
        std::shared_ptr<DiscoveredDeviceUsb> pAoapUsbDevice = DeviceMonitor::instance().getSwitchedUsbDevice();

        if(pAoapUsbDevice != nullptr)
        {
            DiscoveryError res = DiscoveryError::INCOMPLETE;
            int32_t cnt = 0;

            while ((res == DiscoveryError::INCOMPLETE) && (cnt < 8))
            {
                res = pAoapUsbDevice->resetDevice(FD_PROTOCOL_GOOGLE_AOAP, 1000);

                if (DiscoveryError::INCOMPLETE == res)
                {
                    mspin_log_printLn(eMspinVerbosityInfo, "%s() unable to reset device with res = %s",
                             __FUNCTION__, to_str(res));
                }
                else if (DiscoveryError::OK != res)
                {
                    mspin_log_printLn(eMspinVerbosityWarn, "%s() reset device failed with res = %s",
                             __FUNCTION__, to_str(res));
                    break;
                }
                else
                {
                    // don't do anything
                }
                cnt++;
            }
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityVerbose, "%s() pAoapUsbDevice is NULL", __FUNCTION__);
        }
    }

    mspin_log_printLn(eMspinVerbosityDebug,
            "%s(args=%p) [id=%lu] resources released -> exit thread id=%lu now",
            __FUNCTION__, pMyspinInstance, pMyspinInstance->threadId,
            pMyspinInstance->threadId);

    --sActiveConnections;

    delete (pMyspinInstance->pConnection);
    pMyspinInstance->pConnection = NULL;

    if (0 != pthread_equal(pMyspinInstance->threadId, gMyspinInstances[0].threadId))
    {
        gMyspinInstances[0].threadId = 0;
    }
    else if (0 != pthread_equal(pMyspinInstance->threadId, gMyspinInstances[1].threadId))
    {
        gMyspinInstances[1].threadId = 0;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(args=%p) [id=%lu] ERROR: thread id=%lu does not match neither to %lu nor to %lu",
                __FUNCTION__, pMyspinInstance, pMyspinInstance->threadId,
                (unsigned long int)pMyspinInstance->threadId,
                gMyspinInstances[0].threadId, gMyspinInstances[1].threadId);
    }

    // Signal to main thread
    std::unique_lock<std::mutex> lk(g_session_monitor_m);
    g_running_session = true;
    lk.unlock();

    g_session_monitor_cv.notify_one();
    mspin_log_printLn(eMspinVerbosityVerbose, "%s: Session is stopped", __FUNCTION__);

    pthread_exit(NULL);
    return NULL;
}

static bool mspin_demo_initMyspinInstance(uint32_t vendorId, uint32_t productId, std::string serial, uint32_t devNum)
{
    mspin_log_printLn(eMspinVerbosityInfo, "%s() starts", __FUNCTION__);

    // Check status when starting myspin instance
    bool ret = false;

    int tb_time = 0;
    S32 instanceIndex = -1;
    bool res_wait = false;
    mspin_demo_connectionParameter_t* param = new mspin_demo_connectionParameter_t;
    param->pMonitorContext = &monitorContext;

    if(Testbed::instance().getTestbedMode())
    {
        Testbed::instance().adit_testbed_available();
        Testbed::instance().setAndroidPhoneDisconnectionStatus(FALSE);
        Testbed::instance().setApplePhoneDisconnectionStatus(FALSE);
    }

    if (!param)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s() FATAL ERROR: Could not allocate memory for mySPIN connection",
                __FUNCTION__);
        return false;
    }

    if (NULL == gMyspinInstances[0].pConnection)
    {
        instanceIndex = 0;
    }
    else if (NULL == gMyspinInstances[1].pConnection)
    {
        instanceIndex = 1;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(%.4x:%.4x, s/n='%s', devNum=%d) ERROR: Already two connected devices -> unplug first one and then try again",
                __FUNCTION__, vendorId, productId, serial.c_str(), devNum);
        delete param;
        return false;
    }

    gMyspinInstances[instanceIndex].pConnection = param;

    pthread_mutex_init(&(param->switchStateMutex), NULL);
    param->vendorId = vendorId;
    param->productId = productId;

    if (param->serial.max_size() < serial.size())
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(%.4x:%.4x, s/n='%s', devNum=%d) FATAL ERROR: Serial is too long",
                __FUNCTION__, vendorId, productId, serial.c_str(), devNum);
    }

    param->serial = serial;
    param->devNum = devNum;
    param->boardTypeSD = gBoardTypeSD;
    param->runState = eMSPIN_DEMO_STATE_STOPPED;
    param->pMyspin = NULL;

    pthread_mutex_lock(&(param->switchStateMutex));
    if (DeviceMonitor::instance().isIOSDevice(param->vendorId, param->productId))
    {
        param->audioSupport = gEnableAudio;
        param->switchRunState = eMSPIN_DEMO_STATE_STABLE;
    }
    else
    {
        param->audioSupport = gEnableAudio;
        param->switchRunState = eMSPIN_DEMO_STATE_SWITCH_TO_RUN;
    }

    if(gWifiConnect == true)
    {
        MSPIN_StartTCPListener(8159,
                &DeviceMonitor::OnAcceptIncomingConnection_CB,
                &DeviceMonitor::OnConnectionClosed_CB,
                (void*)param);

        MSPIN_UDP_MESSAGE_PARAMETER_t udpConf;
        udpConf.manufacturer = (const U8*)"CAF";
        udpConf.model = (const U8*)"I06";
        udpConf.vendor = (const U8*)"GEN3";
        udpConf.version = (const U8*)"1.0";
        udpConf.tcpPort =8159;

        const U8* interface =  (unsigned const char *)"wlan0";

        MSPIN_StartUDPBroadcasting(8158 ,interface ,udpConf ,2000 , &DeviceMonitor::OnUDPBroadcastEnd_CB,(void*)param);
    }

    pthread_mutex_unlock(&(param->switchStateMutex));
    pthread_attr_t attr;
    pthread_attr_init(&attr);
    pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED); // thread won't be joined; avoids leak

    if(Testbed::instance().getTestbedMode())
    {
        while(tb_time < 4)
        {
            if(Testbed::instance().getAndroidPhoneDisconnectionStatus() ||
                    Testbed::instance().getApplePhoneDisconnectionStatus())
                break;
            tb_time++;
            sleep(1);
        }
    }
    else
    {
        if(!gWifiConnect)
        {
            sleep(3);
        }
    }

    if(gWifiConnect)
    {
        // wait for connection of device via wifi
        std::unique_lock<std::mutex> lk(monitorContext.g_wifi_monitor_m);

        if(gDemoTimeout > 0)
        {
            // Smoketest mode : Having timeout when waiting for signal
            res_wait = monitorContext.g_wifi_monitor_cv.wait_for(lk, std::chrono::seconds(gDemoTimeout),
                       []{return (monitorContext.g_connected_device || monitorContext.g_disconnected_device);});

            if(!res_wait)
            {
                // Time is expired
                mspin_log_printLn(eMspinVerbosityWarn,
                                        "%s: Time is expired %d.", __FUNCTION__, gDemoTimeout);
                monitorContext.g_disconnected_device = true;
                monitorContext.gQuit = true;
                gMSPINDEMO_run = false;
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityWarn,
                                        "%s: Smoketest mode - Signal is received when waiting for connection.", __FUNCTION__);
            }
        }
        else
        {
            // Demo mode : Waiting until receiving signal without timeout.
            monitorContext.g_wifi_monitor_cv.wait(lk, []{return (monitorContext.g_connected_device || monitorContext.g_disconnected_device);});
            mspin_log_printLn(eMspinVerbosityWarn,
                                    "%s: Demo mode - Signal is received when waiting for connection.", __FUNCTION__);
        }

        lk.unlock();

    }
    if (!monitorContext.gQuit)
    {
        if( Testbed::instance().getTestbedMode() &&
                (Testbed::instance().getAndroidPhoneDisconnectionStatus()
                        || Testbed::instance().getApplePhoneDisconnectionStatus()) )
        {
            // release gMyspinInstances
            delete (gMyspinInstances[instanceIndex].pConnection);
            gMyspinInstances[instanceIndex].pConnection = NULL;
            gMyspinInstances[instanceIndex].threadId = 0;
        }
        else
        {
            gDemoAppState = eMSPIN_RETURN_STATE_PHONE_APPEARED;
            pthread_create(&(gMyspinInstances[instanceIndex].threadId), &attr, mspin_demo_runMyspinInstance,
                    &(gMyspinInstances[instanceIndex]));
            pthread_attr_destroy(&attr);

            mspin_log_printLn(eMspinVerbosityDebug,
                    "%s(%.4x:%.4x, s/n='%s', devNum=%d) %s thread with id=%lu created",
                    __FUNCTION__, vendorId, productId, serial.c_str(), devNum,
                    (0 == instanceIndex) ? "first" : "second",
                    gMyspinInstances[instanceIndex].threadId);

            ret = true;
        }
    }

    return ret;
}

#ifndef MSPIN_DEMO_TOUCH_DISABLE
static void* mspin_demo_pumpTouchEvents(void *pArg)
{
    mspin_demo_touch_wl_context_t *pWLContext = (mspin_demo_touch_wl_context_t*)pArg;

    prctl(PR_SET_NAME,"homeToucher",0,0,0);

    while (!monitorContext.gQuit)
    {
        demoTouch.wlPumpEvents(pWLContext, 50); //blocks at max 50ms
    }

    return NULL;
}

static mspin_demo_touch_wl_context_t* mspin_demo_startTouchArea(void)
{
    mspin_log_printLn(eMspinVerbosityInfo,
            "%s() using layerID=%d, surfaceId=%d, offset_x=0, offset_y=0, screen_w=%d, screen_h=%d ",
            __FUNCTION__,
            MSPIN_DEMO_APPL_TOUCH_LAYER_ID, MSPIN_DEMO_APPL_TOUCH_SURFACE_ID,
            MSPIN_DEMO_APPL_SCREEN_WIDTH, MSPIN_DEMO_APPL_SCREEN_HEIGHT);

    //No offset, using layer id=4002 (must be identical to the one used in the start script),
    // surface id=12
    mspin_demo_touch_wl_context_t *pWLContext = demoTouch.touchInit(MSPIN_DEMO_APPL_TOUCH_LAYER_ID,
            MSPIN_DEMO_APPL_TOUCH_SURFACE_ID, 0, 0,
            MSPIN_DEMO_APPL_SCREEN_WIDTH, MSPIN_DEMO_APPL_SCREEN_HEIGHT);

    if (pWLContext)
    {
        if (0 != demoTouch.wlServerInit(pWLContext))
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: Could not init WL Server for touch field",
                    __FUNCTION__);
        }
        else
        {
            if (1 != demoTouch.createImageBuffer(pWLContext,
                    MSPIN_DEMO_APPL_TOUCH_AREA1_WIDTH,
                    MSPIN_DEMO_APPL_TOUCH_AREA1_HEIGHT))
            {
                mspin_log_printLn(eMspinVerbosityError,
                        "%s() ERROR: Could not create image buffer",
                        __FUNCTION__);
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityVerbose,
                        "%s() starting pump events loop",
                        __FUNCTION__);
            }

            //Show one frame to enable touch events (this is mandatory)
            demoTouch.wlUpdateFrameBuffer(pWLContext);

            //Start pump event thread
            if(0 == pthread_create(&gTouchHandleThread, NULL, mspin_demo_pumpTouchEvents, (void*)pWLContext))
            {
                mspin_log_printLn(eMspinVerbosityError, "%s() starting touch handling thread", __FUNCTION__);
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityError,
                        "%s() ERROR: Failed to start touch handling thread",
                        __FUNCTION__);
            }
        }
    }

    return pWLContext;
}

#endif //#ifndef MSPIN_DEMO_TOUCH_DISABLE

void* main_myspin_thread(void* inData)
{
    char writeEndChar = MSPIN_END_CHAR;
    (void)inData;

    usb_device suitableDevice;
    bool startedMonitor = false;
    bool resSessionWait;

    MSPIN_Init();
#ifndef MSPIN_IAP2_SUPPORT_DISABLED
    Iap2Connection::registerDLT();
#endif //#ifndef MSPIN_IAP2_SUPPORT_DISABLED

    if (gEnableFrameMeasurements)
    {
        MSPIN_MeasureFrameLatency();
    }

#ifndef MSPIN_DEMO_TOUCH_DISABLE
    //This code must be placed after 'ilm_init()' which is called in 'MSPIN_Init()'
    mspin_demo_touch_wl_context_t *pWLTouchContext = NULL;
    if (gEnableTouchFields)
    {
        pWLTouchContext = mspin_demo_startTouchArea();
    }
#endif //#ifndef MSPIN_DEMO_TOUCH_DISABLE

    //Wait for devices to appear
    (void)sem_init(&monitorContext.gConnectSemaphore, 0, 0); // lock binary semaphore

    if (!gWifiConnect)
    {
        if(!DeviceMonitor::instance().start_monitor(gDemoTimeout))
        {
            gSmoketestTimeout = 1; //to terminate in 1 second
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: Failed to start monitoring thread", __FUNCTION__);
        }
        else
        {
            startedMonitor = true;
        }

        if (!gSmoketestTimeout)
        {
            while (!monitorContext.gQuit)
            {
                if(DeviceMonitor::instance().getSizeOfUsbDeviceList() == 0)
                {
                    mspin_log_printLn(eMspinVerbosityInfo,
                            "%s() Waiting for device to appear ...\n", __FUNCTION__);
                    (void) sem_wait(&monitorContext.gConnectSemaphore); // wait until device found
                }

                if (DeviceMonitor::instance().getSizeOfUsbDeviceList() > 0) // scan to find the suitable device for running
                {
                    sleep(1);

                    suitableDevice = DeviceMonitor::instance().getFirstItemOfUsbDeviceList();
                    if(mspin_demo_initMyspinInstance(suitableDevice.vendorId, suitableDevice.productId,
                            suitableDevice.serial, suitableDevice.devNum))
                    {
                        // Wait for status of running session
                        std::unique_lock<std::mutex> lk(g_session_monitor_m);

                        if(gDemoTimeout > 0)
                        {
                            resSessionWait = g_session_monitor_cv.wait_for(lk, std::chrono::seconds(gDemoTimeout),
                                                   []{return (g_running_session);});
                            if(!resSessionWait)
                            {
                                mspin_log_printLn(eMspinVerbosityVerbose,
                                        "%s: Time is expired when waiting for session %d.", __FUNCTION__, gDemoTimeout);
                            }
                            else
                            {
                                mspin_log_printLn(eMspinVerbosityVerbose,
                                        "%s: Smoketest mode - Signal is received when waiting for session.", __FUNCTION__);
                            }
                        }
                        else
                        {
                            g_session_monitor_cv.wait(lk, []{return (g_running_session);});

                            mspin_log_printLn(eMspinVerbosityInfo,
                                        "%s: Demo mode - Signal is received when waiting for session.", __FUNCTION__);
                        }

                        lk.unlock();

                        if(g_session_status == MSPIN_SUCCESS)
                        {
                            mspin_log_printLn(eMspinVerbosityInfo,
                                       "%s: Session is running successfully.", __FUNCTION__);
                        }
                        else
                        {
                            // remove the device that can not run mySPIN
                            DeviceMonitor::instance().removeFirstItemOfUsbDeviceList();
                        }

                        g_running_session = false;
                    }
                }
            }
        }
        else
        {
            //Wait 5 seconds in smoketest mode
            gDemoAppState = eMSPIN_RETURN_STATE_SUCCESSFUL_CONNECTION;  // return SUCCESS in case of just wait and exit the demo app
            mspin_log_printLn(eMspinVerbosityInfo, "%s() started in SMOKETEST mode. Waiting %d seconds, then programs terminates itself",
                    __FUNCTION__, gSmoketestTimeout);
            sleep(gSmoketestTimeout);
        }

    }
    else
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s() Wifi initialization", __FUNCTION__);

        struct usb_device gUsbDevice;
        mspin_demo_initMyspinInstance(gUsbDevice.vendorId, gUsbDevice.productId, gUsbDevice.serial, gUsbDevice.devNum);

        while(!monitorContext.gQuit)
        {
            if (gSmoketestTimeout)
            {
                sleep(gSmoketestTimeout);
                break;
            }

            // wait for disconnected device or signal Ctrl-C
            mspin_log_printLn(eMspinVerbosityWarn,
                            "%s: Wait for signal Ctl-C, disconnection %d or time out %d", __FUNCTION__,
                            monitorContext.g_disconnected_device, gSmoketestTimeout);

            std::unique_lock<std::mutex> lk(monitorContext.g_wifi_monitor_m);
            monitorContext.g_wifi_monitor_cv.wait(lk, []{return (monitorContext.g_disconnected_device);});
            lk.unlock();

            break; // due to reconnection feature is not available
        }
        sleep(4);
    }

    // Write end character to the pipe
    if (write(gEndCharFd[1], &writeEndChar, sizeof(char)) != 1)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(): write: %d %s", __FUNCTION__, errno, strerror(errno));
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s() Terminating...", __FUNCTION__);

    if (startedMonitor)
    {
        monitorContext.gQuit = TRUE;
        DeviceMonitor::instance().stop_monitor();
    }

#ifndef MSPIN_DEMO_TOUCH_DISABLE
    if (0 != gTouchHandleThread)
    {
        mspin_log_printLn(eMspinVerbosityDebug,
                "%s() join mspin_demo_pumpTouchEvents thread with id=%d ...",
                __FUNCTION__, (S32)gTouchHandleThread);
        pthread_join(gTouchHandleThread, NULL);
        gTouchHandleThread = 0;

        if (pWLTouchContext)
        {
            demoTouch.touchFinalize(pWLTouchContext);
            pWLTouchContext = NULL;
        }
    }
#endif //#ifndef MSPIN_DEMO_TOUCH_DISABLE

    //wait 3 second to be sure the MSPIN_DeleteInstance() has been finished and
    //wait for switch USB port configuration
    sleep(3);

    MSPIN_Shutdown();
#ifndef MSPIN_IAP2_SUPPORT_DISABLED
    Iap2Connection::unregisterDLT();
#endif //#ifndef MSPIN_IAP2_SUPPORT_DISABLED

    mspin_log_printLn(eMspinVerbosityInfo, "%s() BYE!", __FUNCTION__);

    return nullptr;
}

int myspin_demo_registerSignal()
{
    int sfd;
    sigset_t mask;

    sigemptyset(&mask);
    sigaddset(&mask, SIGINT);
    sigaddset(&mask, SIGQUIT);
    sigaddset(&mask, MSPIN_KEY_HOME_SIGNO);
    sigaddset(&mask, MSPIN_KEY_BACK_SIGNO);
    sigaddset(&mask, MSPIN_KEY_MENU_SIGNO);
    sigaddset(&mask, MSPIN_KEY_SEARCH_SIGNO);

    // Listen to SIGALRM only when timeout is set
    if (0 != gDemoTimeout)
    {
        sigaddset(&mask, SIGALRM);
    }

    if (sigprocmask(SIG_BLOCK, &mask, NULL) != 0)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() sigprocmask: %d %s", __FUNCTION__, errno, strerror(errno));
        return -1;
    }

    sfd = signalfd(-1, &mask, SFD_NONBLOCK|SFD_CLOEXEC);

    if (sfd == -1)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() signalfd: %d %s \n", __FUNCTION__, errno, strerror(errno));
    }
    return sfd;
}


int main(int argc, char *argv[])
{
    pthread_t mySPIN_thread;

    int c;
    int verbosityLevel = -1;
    int longoptsIndex = 0;
    bool enableTimeStamps = FALSE;
    int sfd;
    struct signalfd_siginfo fdsi;
    ssize_t s;
    fd_set rfds;
    int retVal;
    char readEndChar;

    monitorContext.gQuit = FALSE;
    monitorContext.g_connected_device = FALSE;
    monitorContext.g_disconnected_device = FALSE;

    signal(SIGPIPE, SIG_IGN);   // avoid termination on SIGPIPE by libusb

    //Reset myspin instance values
    gMyspinInstances[0].threadId = 0;
    gMyspinInstances[0].pConnection = NULL;

    gMyspinInstances[1].threadId = 0;
    gMyspinInstances[1].pConnection = NULL;

    while ((c = getopt_long(argc, argv, "a:c:b:dfg:hi:k:pr::s::tu:v:w:xz:", longopts, &longoptsIndex)) != -1)
    {
        switch (c)
        {
            case 'h':
            {
                fprintf(stdout, "USAGE: mspin_demo_appl [-v <level>] [-a0|1|2] [-b ai|sd] [-c <hw>] [-i <layerid>] [-k0|1|2] [-p] [-r[<hw>]] [-s[timeout]] [-t] [-u<height>] [-v<level>] [-w <width>]\n");
                fprintf(stdout, "USAGE: mspin_demo_appl -h\n\n");
                fprintf(stdout, "Options:\n");
                fprintf(stdout, "  -a[0|1|2]   : 0 = Disable, 1 = Enable audio(old implementation),2 = Enable mySPINAudioStreamLib (default disabled)\n");
                fprintf(stdout, "  -bai|sd     : ai = ai board, sd = sd board (default sd)\n");
                fprintf(stdout, "  -c<hw>      : sets the audio sink (default is '%s'). Only relevant when audio is enabled\n", gAudioSink);
                fprintf(stdout, "  -f          : enable frame measurements. This feature must be also enabled per compile flag 'MSPIN_ENABLE_FRAME_MEASUREMENTS'\n");
                fprintf(stdout, "  -i          : layer id of the (first) mySPIN instance\n");
                fprintf(stdout, "  -k[0|1|2]   : frame compression (0 = no compression, 1 = zlib (default), 2 = JPEG\n");
                fprintf(stdout, "  -p          : enable performance measurements for AOAP read and write\n");
                fprintf(stdout, "  -r[<hw>]    : sets the audio source. If no additional value is specified the default='%s' will be used. If not specified\n", gAudioSource);
                fprintf(stdout, "                the audio source will be automatically detected\n");
                fprintf(stdout, "  -s[timeout] : enable smoketest mode. A timeout before exiting mySPIN demo can be optionally specified. No mySPIN connection will be established!\n");
                fprintf(stdout,"   -z[timeout] : set smoketest timeout. A timeout before exiting mySPIN demo can be optionally specified. mySPIN connection will be established!\n");
                fprintf(stdout, "  -t          : add a timestamp before every printed debug message (only applicable in addition with option '-v')\n");
                fprintf(stdout, "  -u          : layer height for the (first) mySPIN instance\n");
                fprintf(stdout, "  -v<level>   : specifies verbosity level. Range 0..7\n");
                fprintf(stdout, "  -w          : layer width for the (first) mySPIN instance\n");
                fprintf(stdout, "  -d          : enable Testbed mode\n");
                fprintf(stdout, "  -h          : prints this help\n");
                return 0;
                //break;
            }
            case 'a':
            {
                gEnableAudio = static_cast<U32> (atoi(optarg));
                if(gEnableAudio > 2)
                {
                    fprintf(stdout, "ERROR: Invalid option argument %s for option -a -> ABORT\n", optarg);
                    gEnableAudio=0;
                }
                break;
            }
            case 'b':
            {
                if (0 == strcmp("ai", optarg))
                {
                    gBoardTypeSD = FALSE;
                }
                else if (0 == strcmp("sd", optarg))
                {
                    gBoardTypeSD = TRUE;
                }
                else
                {
                    fprintf(stdout, "ERROR: Invalid option argument %s for option -b -> ABORT\n", optarg);
                    return 0;
                }
                break;
            }
            case 'c':
            {
                if (sizeof(gAudioSink) > strlen(optarg)+1)
                {
                    strncpy(gAudioSink, optarg, strlen(optarg)+1);
                }
                else
                {
                    fprintf(stdout, "ERROR: audio sink '%s' is invalid\n", optarg);
                    return 0;
                }
                break;
            }
            case 'd':
            {
                Testbed::instance().setTestbedMode(TRUE);
                break;
            }
            case 'f':
            {
                gEnableFrameMeasurements = TRUE;
                break;
            }
            case 'i': //id of layer 1
            {
                gLayer1Id = atoi(optarg);
                break;
            }
            case 'k': //frame compression
            {
                if (optarg)
                {
                    if (0 == atoi(optarg))
                        gFrameCompression = MSPIN_FRAME_COMPRESSION_NONE;
                    else if (1 == atoi(optarg))
                        gFrameCompression = MSPIN_FRAME_COMPRESSION_ZLIB;
                    else if (2 == atoi(optarg))
                        gFrameCompression = MSPIN_FRAME_COMPRESSION_JPEG;
                    else
                        fprintf(stdout, "ERROR: compression %s is not supported\n", optarg);
                }
                else
                    fprintf(stdout, "ERROR: Please specify a compression\n");
                break;
            }
            case 'p':
            {
                gEnablePerformanceMeasurements = TRUE;
                break;
            }
            case 'r':
            {
                gDetectAudioSource = FALSE;
                if (optarg)
                {
                    strncpy(gAudioSource, optarg, strlen(optarg)+1);
                    fprintf(stdout, "    Using audio source '%s'\n", gAudioSource);
                }
                else
                {
                    fprintf(stdout, "    Using default audio source '%s'\n", gAudioSource);
                }
                break;
            }
            case 's':
            {
                if (optarg)
                {
                    gSmoketestTimeout = atoi(optarg);
                }
                else
                {
                    gSmoketestTimeout = 5; //5 seconds is default
                }
                break;
            }
            case 'z':
            {
                if (optarg)
                {
                    gDemoTimeout = atoi(optarg);
                }
                else
                {
                    gDemoTimeout = 0; // no timeout is set
                }
                break;
            }
            case 't':
            {
                enableTimeStamps = TRUE;
                break;
            }
            case 'u': //height of layer 1
            {
                gLayer1Height = atoi(optarg);
                break;
            }
            case 'v': //set verbosity level
            {
                verbosityLevel = atoi(optarg);
                break;
            }
            case 'w': //width of layer 1
            {
                gLayer1Width = atoi(optarg);
                break;
            }
            case 'x':
            {
                gWifiConnect = TRUE;
                break;
            }
            default:
            {
                fprintf(stdout, "ERROR: Invalid option %c specified\n", optopt);
                break;
            }
        }
    }

    // Setup the shared variables
    DeviceMonitor::instance().setTestbedInstance(&Testbed::instance());
    DeviceMonitor::instance().setMonitorContext(&monitorContext);

    sfd = myspin_demo_registerSignal();

    //Set debug print level for mySPIN
    if (-1 != verbosityLevel)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s() Disable DLT logging and use level=%d",
                __FUNCTION__, verbosityLevel);
        MSPIN_SetLoggingDestination(MSPIN_LOGGING_TO_STDOUT, true);
        MSPIN_SetVerbosity(verbosityLevel, enableTimeStamps);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s() Use DLT logging", __FUNCTION__);
    }

    //Signal SIGALRM when the timeout is expired
    if (0 != gDemoTimeout)
    {
        // Create timer
        struct itimerspec its;
        struct sigevent sev;
        timer_t timerid = NULL;

        sev.sigev_notify = SIGEV_SIGNAL;
        sev.sigev_signo = SIGALRM;
        sev.sigev_value.sival_ptr = NULL;

        if (timer_create(CLOCK_REALTIME, &sev, &timerid) == -1)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(): timer_create %d %s",
                    __FUNCTION__, errno, strerror(errno));
            return (errno);
        }

        its.it_value.tv_sec     = gDemoTimeout;
        its.it_value.tv_nsec    = 0;
        its.it_interval.tv_sec  = 0;
        its.it_interval.tv_nsec = 0;

        mspin_log_printLn(eMspinVerbosityError, "Automatically exit mySPIN test after %ld seconds", gDemoTimeout);

        if (timer_settime(timerid, 0, &its, nullptr) == -1)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(): timer_settime %d %s",
                    __FUNCTION__, errno, strerror(errno));
            return(errno);
        }
    }

    MSPIN_CountFramesPerSecond(); //activate fps counting

    /* PRQA: Lint Message 717: Ignore. Not in this source code */
    /*lint -save -e717*/
    DLT_REGISTER_APP(DLT_MYSPIN_APP_APID, "mySPIN Demo");
    /*lint -restore*/

    if (pipe (gEndCharFd) == -1)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(): pipe %d %s", __FUNCTION__, errno, strerror(errno));
        return (errno);
    }

    //Create thread for mySPIN
    pthread_create(&mySPIN_thread, NULL, main_myspin_thread, (void*)NULL);

    if(Testbed::instance().getTestbedMode())
    {
        Testbed::instance().adit_testbed_init();
        Testbed::instance().adit_testbed_run();
    }

    //Listening to the hardkey signals and end char pipe
    if (sfd != -1)
    {
        while (true)
        {
            FD_ZERO(&rfds);
            FD_SET(sfd, &rfds);
            FD_SET(gEndCharFd[0], &rfds);

            retVal = select(FD_SETSIZE, &rfds, NULL, NULL, NULL);
            if (retVal == -1)
            {
                mspin_log_printLn(eMspinVerbosityError, "%s(): select: %d %s", __FUNCTION__, errno, strerror(errno));
            }
            else if (retVal != 0)
            {
                if (FD_ISSET(sfd, &rfds))
                {
                    s = read(sfd, &fdsi, sizeof(struct signalfd_siginfo));
                    if (s != sizeof(struct signalfd_siginfo))
                    {
                        mspin_log_printLn(eMspinVerbosityError, "%s(): read signal: %d %s", __FUNCTION__, errno, strerror(errno));
                    }
                    else
                    {
                        mspin_demo_signalHandler(fdsi.ssi_signo);
                    }
                }

                if (FD_ISSET(gEndCharFd[0], &rfds))
                {
                    s = read(gEndCharFd[0], &readEndChar, sizeof(char));
                    if (s != sizeof(char))
                    {
                        mspin_log_printLn(eMspinVerbosityError, "%s(): read end char: %d %s", __FUNCTION__, errno, strerror(errno));
                    }
                    else
                    {
                        if (readEndChar == MSPIN_END_CHAR)
                        {
                            // quit from the loop when myspin_thread ends
                            break;
                        }
                        else
                        {
                            mspin_log_printLn(eMspinVerbosityError, "%s(): read unexpected character", __FUNCTION__);
                        }
                    }
                }
            }
        }
    }

    if (sfd != -1)
    {
        close(sfd);
    }

    pthread_join(mySPIN_thread, NULL);

    return gDemoAppState;
}
